function segmentedImage = segmentImageLDA(imgPath, fgSample, bgSample)
%SEGMENTIMAGELDA Segment image using 1-pass LDA
%
%   Input:
%       imgPath : path to the input image
%       fgSample: pixels of foreground scribbles
%       bgSample: pixels of background scribbles
% 
%   Output:
%       segmentedImage: binary image representing the output segmentation
%
% (c) Moustafa Meshry, moustafa.meshry@alexu.edu.eg
%     Department of Compter and Systems Engineering, Alexandria University, Egypt.


    % extract features for all pixels
    imgFeatures = extractFeatures(imgPath, fgSample, bgSample);
    [height, width, featureVectorSize] = size(imgFeatures);
    
    % Create user fg/bg scribbles binary mask
    fgScribblesMask = false(height, width);
    bgScribblesMask = false(height, width);
    fgScribblesMask(sub2ind(size(fgScribblesMask), ...
        fgSample(:, 1), fgSample(:, 2))) = true;
    bgScribblesMask(sub2ind(size(bgScribblesMask), ...
        bgSample(:, 1), bgSample(:, 2))) = true;
    
    % select features of user-labeled foreground
    fgSampleMask3D = repmat(fgScribblesMask, 1, featureVectorSize);
    fgSampleMask3D = reshape(fgSampleMask3D, size(imgFeatures));
    fgFeatures = imgFeatures(fgSampleMask3D);
    fgFeatures = reshape(fgFeatures, [], featureVectorSize);

    % select features of user-labeled background
    bgSampleMask3D = repmat(bgScribblesMask, 1, featureVectorSize);
    bgSampleMask3D = reshape(bgSampleMask3D, size(imgFeatures));
    bgFeatures = imgFeatures(bgSampleMask3D);
    bgFeatures = reshape(bgFeatures, [], featureVectorSize);

    % LDA Training
    trainSample = [fgFeatures; bgFeatures];
    trainLabels = ones(size(trainSample, 1), 1);
    trainLabels(size(fgFeatures, 1) + 1 : end) = 0;

    W = LDA(trainSample, trainLabels);
    
    % LDA projection and classification
    pixelsVec = reshape(imgFeatures, height * width, []);
    pixelsVec = [ones(size(pixelsVec, 1), 1) pixelsVec];
    pixelsScores = pixelsVec * W';

     % reshape pixelsScores to be of size N x M x 2 (the 2 represents the
     % scores of background and foreground classes)
    pixelsScores = reshape(pixelsScores, ...
        [height, width, size(pixelsScores, 2)]);
    
    % Pixel Classification: select learned background pixels
    bgMask = pixelsScores(:, :, 1) > pixelsScores(:, :, 2);

    % Color background pixels for output
    [y, x] = find(bgMask);
    segmentedImage = repmat(255, height, width);
    segmentedImage(sub2ind(size(segmentedImage), y, x)) = 0;
    segmentedImage = postProcess(segmentedImage, fgScribblesMask, ...
        bgScribblesMask);
end

